#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/capability.h>
#include <signal.h>

static void boing(int sig) {
    printf("*(boing!)*\n");
}

static void drop_caps_be_normal(void) {
    cap_t none;
    /* cap_init() guarantees all caps are cleared */
    if ((none = cap_init()) == NULL)
        fprintf(stderr,"cap_init() failed, aborting...\n");
    if (cap_set_proc(none) == -1) {
        cap_free(none);
        fprintf(stderr,"cap_set_proc('none') failed, aborting...\n");
    }
    cap_free(none);
    /* Become your normal true self again! */
    if (setuid(getuid()) < 0)
        fprintf(stderr,"setuid to lower privileges failed, aborting..\n");
}

static void test_setuid(void) {
    printf("%s:\nRUID = %d EUID = %d\n", __FUNCTION__, getuid(), geteuid());
    if (seteuid(0) == -1)
        printf("seteuid(0) failed...\n");
    printf("RUID = %d EUID = %d\n", getuid(), geteuid());
}

static void usage(char **argv, int stat) {
    fprintf(stderr, "Usage: %s 1|2\n"
        " 1 : add just one capability - CAP_SETUID\n"
        " 2 : add two capabilities - CAP_SETUID and CAP_SYS_ADMIN\n"
        "Tip: run it in the background so that capsets can be looked up\n",
        argv[0]);
    exit(stat);
}

int main(int argc, char **argv) {
    int opt, ncap;
    cap_t mycaps;
    cap_value_t caps2set[2];

    if (argc < 2)
        usage(argv, EXIT_FAILURE);

    opt = atoi(argv[1]);
    if (opt != 1 && opt != 2)
        usage(argv, EXIT_FAILURE);

    if (signal(SIGINT, boing) == SIG_ERR)
        fprintf(stderr,"signal() SIGINT failed, aborting...\n");
    if (signal(SIGTERM, boing) == SIG_ERR)
        fprintf(stderr,"signal() SIGTERM failed, aborting...\n");

    if (!CAP_IS_SUPPORTED(CAP_SETUID))
        fprintf(stderr,("CAP_SETUID capability not supported on system, aborting...\n");
    if (opt == 2) {
        if (!CAP_IS_SUPPORTED(CAP_SYS_ADMIN))
            fprintf(stderr,
                ("CAP_SYS_ADMIN capability not supported on system, aborting...\n");
    }

    mycaps = cap_get_proc();
    if (!mycaps)
        fprintf(stderr,"cap_get_proc() for CAP_SETUID failed, aborting...\n");

    if (opt == 1) {
        ncap = 1;
        caps2set[0] = CAP_SETUID;
    } else if (opt == 2) {
        ncap = 2;
        caps2set[0] = CAP_SETUID;
        caps2set[1] = CAP_SYS_ADMIN;
    }
    if (cap_set_flag(mycaps, CAP_EFFECTIVE, ncap, caps2set, CAP_SET) == -1) {
        cap_free(mycaps);
        fprintf(stderr,"cap_set_flag() failed, aborting...\n");
    }

    if (opt == 1) {
        caps2set[0] = CAP_SYS_ADMIN;
        if (cap_set_flag(mycaps, CAP_EFFECTIVE, 1, caps2set, CAP_CLEAR) == -1) {
            cap_free(mycaps);
            fprintf(stderr,
                ("cap_set_flag(clear CAP_SYS_ADMIN) failed, aborting...\n");
        }
    }

    if (cap_set_proc(mycaps) == -1) {
        cap_free(mycaps);
        fprintf(stderr,
            ("cap_set_proc(CAP_SETUID/CAP_SYS_ADMIN) failed, aborting...\n",
             (opt == 1 ? "CAP_SETUID" : "CAP_SETUID,CAP_SYS_ADMIN"));
    }
    if (opt == 1)
        printf("PID %6d now has CAP_SETUID capability.\n", getpid());
    else if (opt == 2)
        printf("PID %6d now has CAP_SETUID,CAP_SYS_ADMIN capability.\n", getpid());

    printf("Pausing #1 ...\n");
    pause();
    test_setuid();
    cap_free(mycaps);

    printf("Now dropping all capabilities and reverting to original self...\n");
    drop_caps_be_normal();
    test_setuid();

    printf("Pausing #2 ...\n");
    pause();
    printf(".. done, exiting.\n");
    exit(EXIT_SUCCESS);
}
