#include <stdio.h>
#include <unistd.h>
#include <pthread.h>
#include <stdlib.h>
#include <signal.h>

#define HZ  250
#define DELAY_LOOP(val,loop_count)                                             \
{                                                                              \
    int c=0, m;                                                            \
    unsigned int for_index,inner_index;                                    \
                                                                           \
    for(for_index=0;for_index<loop_count;for_index++) {                    \
        beep((val));                                                   \
        c++;                                                           \
        for(inner_index=0;inner_index<HZ*1000;inner_index++)           \
            for(m=0;m<30;m++);                                     \
        }                                                              \
    /*printf("c=%d\n",c);*/                                                \
}


#define NTHREADS     2
#define NUMWORK    200

/* Run with:
 * scheduling policy : SCHED_RR
 * realtime priority : rt_prio
 */
void *worker1(void *msg) {
    struct sched_param p;
    /* The structure used is defined in linux/sched.h as:
     * struct sched_param {
     *      int sched_priority;
     * };
     */

    printf("  RT Thread p1 (%s():%d:LWP %d):\n"
           " Setting sched policy to SCHED_RR and RT priority to %ld and sleeping for 2s ...\n",
           __func__, __LINE__, getpid(), (long)msg);

    /* pthread_setschedparam(3) internally invokes the syscall
     * sched_setscheduler(2).  */
    p.sched_priority = (long)msg;
    if (pthread_setschedparam(pthread_self(), SCHED_RR, &p))
        printf("pthread_setschedparam failed\n");

    sleep(2);

    puts("  p1 working");
    DELAY_LOOP('1', NUMWORK);

    puts("  p1: exiting..");
    pthread_exit((void *)0);
}

/* Run with:
 * scheduling policy : SCHED_FIFO
 * realtime priority : rt_prio + 10
 */
void *worker2(void *msg) {
    struct sched_param p;
    /* The structure used is defined in linux/sched.h as:
     * struct sched_param {
     *      int sched_priority;
     * };
     */
    long prio = (long)msg;

    printf("  RT Thread p2 (%s():%d:LWP %d):\n"
           " Setting sched policy to SCHED_FIFO and RT priority to %ld and sleeping for 4s ...\n",
           __func__, __LINE__, getpid(), (long)msg+10);

    /* pthread_setschedparam(3) internally invokes the syscall
     * sched_setscheduler(2).  */
    p.sched_priority = prio + 10;
    if (pthread_setschedparam(pthread_self(), SCHED_FIFO, &p))
        printf("pthread_setschedparam failed\n");
    sleep(4);

    puts("  p2 working");
    DELAY_LOOP('2', NUMWORK);

    puts("  p2 exiting ...");
    pthread_exit((void *)0);
}

int main(int argc, char **argv) {
    pthread_t tid[NTHREADS];
    pthread_attr_t attr;
    int ret, min, max;
    long rt_prio = 1;

    if (argc == 1) {
        fprintf(stderr, "Usage: %s realtime-priority\n", argv[0]);
        exit(EXIT_FAILURE);
    }

    min = sched_get_priority_min(SCHED_FIFO);
    if (min == -1)
        fprintf(stderr, "sched_get_priority_min failure");
    max = sched_get_priority_max(SCHED_FIFO);
    if (max == -1)
        fprintf(stderr, "sched_get_priority_max failure");
    printf("SCHED_FIFO: priority range is %d to %d\n", min, max);

    rt_prio = atoi(argv[1]);
    if ((rt_prio < min) || (rt_prio > (max - 10)))
        fprintf(stderr, "%s: Priority value passed (%ld) out of range [%d-%d].\n",
              argv[0], rt_prio, min, (max - 10));

    if (geteuid() != 0)
        printf("\nNote: to create true RT threads, you need to run this"
               " program as superuser\n");

    /* Create the two (soft) RT worker threads */
    pthread_attr_init(&attr);
    pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);

    printf("main: creating RT worker thread #1 ...\n");
    ret = pthread_create(&tid[0], &attr, worker1, (void *)rt_prio);
    if (ret)
        fprintf(stderr, "pthread_create() #1 failed! [%d]\n", ret);

    printf("main: creating RT worker thread #2 ...\n");
    ret = pthread_create(&tid[1], &attr, worker2, (void *)rt_prio);
    if (ret)
        fprintf(stderr, "pthread_create() #2 failed! [%d]\n", ret);

    pthread_attr_destroy(&attr);

    DELAY_LOOP('m', NUMWORK+100);

    printf("\nmain: all done, app exiting ...\n");
    pthread_exit((void *)0);
}